﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/DescribeDBClustersMessage">AWS
 * API Reference</a></p>
 */
class DescribeDBClustersRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeDBClustersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDBClusters"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The user-supplied DB cluster identifier or the Amazon Resource Name (ARN) of
   * the DB cluster. If this parameter is specified, information for only the
   * specific DB cluster is returned. This parameter isn't case-sensitive.</p>
   * <p>Constraints:</p> <ul> <li> <p>If supplied, must match an existing DB cluster
   * identifier.</p> </li> </ul>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  DescribeDBClustersRequest& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that specifies one or more DB clusters to describe.</p> <p>Supported
   * Filters:</p> <ul> <li> <p> <code>clone-group-id</code> - Accepts clone group
   * identifiers. The results list only includes information about the DB clusters
   * associated with these clone groups.</p> </li> <li> <p>
   * <code>db-cluster-id</code> - Accepts DB cluster identifiers and DB cluster
   * Amazon Resource Names (ARNs). The results list only includes information about
   * the DB clusters identified by these ARNs.</p> </li> <li> <p>
   * <code>db-cluster-resource-id</code> - Accepts DB cluster resource identifiers.
   * The results list will only include information about the DB clusters identified
   * by these DB cluster resource identifiers.</p> </li> <li> <p> <code>domain</code>
   * - Accepts Active Directory directory IDs. The results list only includes
   * information about the DB clusters associated with these domains.</p> </li> <li>
   * <p> <code>engine</code> - Accepts engine names. The results list only includes
   * information about the DB clusters for these engines.</p> </li> </ul>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeDBClustersRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeDBClustersRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so you can retrieve the remaining
   * results.</p> <p>Default: 100</p> <p>Constraints: Minimum 20, maximum 100</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeDBClustersRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous
   * <code>DescribeDBClusters</code> request. If this parameter is specified, the
   * response includes only records beyond the marker, up to the value specified by
   * <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeDBClustersRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the output includes information about clusters shared from
   * other Amazon Web Services accounts.</p>
   */
  inline bool GetIncludeShared() const { return m_includeShared; }
  inline bool IncludeSharedHasBeenSet() const { return m_includeSharedHasBeenSet; }
  inline void SetIncludeShared(bool value) {
    m_includeSharedHasBeenSet = true;
    m_includeShared = value;
  }
  inline DescribeDBClustersRequest& WithIncludeShared(bool value) {
    SetIncludeShared(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBClusterIdentifier;

  Aws::Vector<Filter> m_filters;

  int m_maxRecords{0};

  Aws::String m_marker;

  bool m_includeShared{false};
  bool m_dBClusterIdentifierHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_includeSharedHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
